using System;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class is a datatype encapsulating validation for RPC parameter position.
	/// </summary>
	public class ParameterPosition
	{
		int _position;

		/// <summary>
		/// Creates RPC parameter position from the supplied 
		/// integer perfoming all necessary validations.
		/// </summary>
		/// <param name="parameterPosition">An integer representing RPC parameter position.</param>
		public ParameterPosition( int parameterPosition )
		{
			if( parameterPosition < 1 || parameterPosition > GlobalConfig.MaxRpcParametersNumber )
				throw( new ArgumentOutOfRangeException( "parameterPosition", SR.Exceptions.InvalidRpcParameterPosition( 1, GlobalConfig.MaxRpcParametersNumber, parameterPosition ) ) );

			_position = parameterPosition;
		}

		/// <summary>
		/// Parsing method creating an instance of RPC parameter position from supplied string.
		/// </summary>
		/// <param name="parameterPositionString">String to parse.</param>
		/// <returns>Newly created instance of RPC parameter position.</returns>
		public static ParameterPosition Parse( string parameterPositionString )
		{
			if( parameterPositionString == null )
				throw( new ArgumentNullException( "parameterPositionString" ) );

			try
			{
				return new ParameterPosition( Int32.Parse( parameterPositionString ) );
			}
			catch( ArgumentOutOfRangeException xcp )
			{
				throw( new StringParseException( xcp.Message ) );
			}
			catch( Exception xcp ) // have to catch System.Exception here because Int32.Parse throws 3 different exceptions. 
			{
				// Rethrowing if this is not exception caused by Int32.Parse 
				// (ArgumentNullException cannot occur) or ParameterPosition constructor
				if( !( xcp is FormatException || xcp is OverflowException || xcp is ArgumentException ) )
					throw;

				throw( new StringParseException( SR.Exceptions.InvalidRpcParameterPosition( 1, GlobalConfig.MaxRpcParametersNumber, parameterPositionString ), xcp ) );
			}
		}

		/// <summary>
		/// Conversion method returning integer representation of RPC parameter position.
		/// </summary>
		/// <returns>RPC parameter position as integer.</returns>
		public int ToInt32()
		{
			return _position;
		}

		/// <summary>
		/// Conversion method returning string representation of RPC parameter position.
		/// </summary>
		/// <returns>RPC parameter position as string.</returns>
		public override string ToString()
		{
			return _position.ToString();
		}
	}
}
